-- ============================================
-- ANÁLISIS DE RELACIONES Y CORRECCIONES
-- ============================================

SET search_path TO dulce_control, public;

-- ============================================
-- CORRECCIÓN: detalles_plan_produccion necesita mejor soporte para personalizaciones
-- ============================================

ALTER TABLE detalles_plan_produccion 
    ADD COLUMN IF NOT EXISTS es_personalizado BOOLEAN DEFAULT FALSE,
    ADD COLUMN IF NOT EXISTS detalle_pedido_id BIGINT REFERENCES detalles_pedido(id) ON DELETE SET NULL;

COMMENT ON COLUMN detalles_plan_produccion.es_personalizado IS 
'Marca si este ítem requiere personalización especial. Ayuda al panadero a identificar productos que necesitan atención extra';

COMMENT ON COLUMN detalles_plan_produccion.detalle_pedido_id IS 
'Referencia directa al detalle del pedido (que contiene la personalización). Útil para pedidos futuros con personalización';

-- ============================================
-- ÍNDICES BASE (UBIGEO)
-- ============================================

CREATE INDEX IF NOT EXISTS idx_ubigeo_provincias_dpto_id 
    ON ubigeo_provincias(departamento_id);

CREATE INDEX IF NOT EXISTS idx_ubigeo_distritos_prov_id 
    ON ubigeo_distritos(provincia_id);

-- ============================================
-- ÍNDICES ORGANIZADOS POR MÓDULO
-- ============================================

-- ==============================
-- ÍNDICES PARA SUPERADMINISTRADOR
-- ==============================

-- Dashboard: KPIs en tiempo real
CREATE INDEX IF NOT EXISTS idx_tiendas_estado_activo 
    ON tiendas(estado) 
    WHERE eliminado_en IS NULL;

CREATE INDEX IF NOT EXISTS idx_suscripciones_estado_fecha 
    ON suscripciones(estado, fecha_fin) 
    WHERE estado IN ('activa', 'vencida');

CREATE INDEX IF NOT EXISTS idx_comprobantes_sunat_estado 
    ON comprobantes(estados_sunat, fecha_emision DESC) 
    WHERE estados_sunat IN ('observado', 'rechazado');

-- Gestión de Tiendas
CREATE INDEX IF NOT EXISTS idx_tiendas_busqueda 
    ON tiendas(numero_doc, nombre_comercial);

CREATE INDEX IF NOT EXISTS idx_sedes_tienda_activas 
    ON sedes(tienda_id, activo) 
    WHERE eliminado_en IS NULL;

CREATE INDEX IF NOT EXISTS idx_dominios_tienda 
    ON dominios_tienda(tienda_id, tipo);

CREATE INDEX IF NOT EXISTS idx_usuarios_tienda_activos 
    ON usuarios_tienda(tienda_id, activo) 
    WHERE eliminado_en IS NULL;

-- Planes y Suscripciones
CREATE INDEX IF NOT EXISTS idx_planes_activos 
    ON planes(activo, codigo);

CREATE INDEX IF NOT EXISTS idx_suscripciones_tienda 
    ON suscripciones(tienda_id, estado, fecha_fin DESC);

CREATE INDEX IF NOT EXISTS idx_historial_suscripciones_lookup 
    ON historial_suscripciones(suscripcion_id, fecha_movimiento DESC);

-- Facturación y SUNAT
CREATE INDEX IF NOT EXISTS idx_comprobantes_tienda_fecha 
    ON comprobantes(tienda_id, fecha_emision DESC);

CREATE INDEX IF NOT EXISTS idx_comprobantes_tipo_serie 
    ON comprobantes(tipos_comprobante, serie_id, correlativo DESC);

CREATE INDEX IF NOT EXISTS idx_transacciones_comprobante 
    ON transacciones_pago(comprobante_id, estado);

CREATE INDEX IF NOT EXISTS idx_transacciones_pasarela 
    ON transacciones_pago(pasarela, estado, creado_en DESC);

CREATE INDEX IF NOT EXISTS idx_detalles_comprobante_factura 
    ON detalles_comprobante(comprobante_id);

-- Centro de Soporte
CREATE INDEX IF NOT EXISTS idx_tickets_estado_prioridad 
    ON tickets_soporte(estado, prioridad, creado_en DESC);

CREATE INDEX IF NOT EXISTS idx_tickets_asignacion 
    ON tickets_soporte(asignado_a_id, estado) 
    WHERE estado NOT IN ('resuelto', 'cerrado');

CREATE INDEX IF NOT EXISTS idx_mensajes_ticket_orden 
    ON mensajes_ticket(ticket_id, creado_en);

-- Sistema y Seguridad
CREATE INDEX IF NOT EXISTS idx_actividad_superadmin_fecha 
    ON actividad_superadmin(admin_id, creado_en DESC);

CREATE INDEX IF NOT EXISTS idx_usuarios_superadmin_correo 
    ON usuarios_superadmin(correo) 
    WHERE activo = TRUE AND eliminado_en IS NULL;

-- ==============================
-- ÍNDICES PARA ADMINISTRADOR
-- ==============================

-- Dashboard
CREATE INDEX IF NOT EXISTS idx_pedidos_tienda_fecha 
    ON pedidos(tienda_id, creado_en DESC);

CREATE INDEX IF NOT EXISTS idx_pedidos_estado_pago 
    ON pedidos(tienda_id, estado_pedido, estado_pago) 
    WHERE estado_pedido != 'cancelado';

-- Clientes
CREATE INDEX IF NOT EXISTS idx_clientes_tienda_busqueda 
    ON clientes(tienda_id, numero_doc, email) 
    WHERE activo = TRUE;

CREATE INDEX IF NOT EXISTS idx_clientes_usuario_virtual 
    ON clientes(tienda_id, es_usuario_virtual) 
    WHERE es_usuario_virtual = TRUE;

CREATE INDEX IF NOT EXISTS idx_direcciones_cliente 
    ON direcciones_cliente(cliente_id, es_entrega);

-- Ventas & Pedidos (POS)
CREATE INDEX IF NOT EXISTS idx_pedidos_borrador_pos 
    ON pedidos(sede_origen_id, vendedor_id, estado_pedido) 
    WHERE estado_pedido = 'borrador';

CREATE INDEX IF NOT EXISTS idx_pedidos_sede_fecha 
    ON pedidos(sede_origen_id, estado_pedido, creado_en DESC);

CREATE INDEX IF NOT EXISTS idx_detalles_pedido_producto 
    ON detalles_pedido(pedido_id, producto_id);

CREATE INDEX IF NOT EXISTS idx_direcciones_pedido_envio 
    ON direcciones_pedido(pedido_id);

CREATE INDEX IF NOT EXISTS idx_pagos_pedido 
    ON pagos_pedido(pedido_id, fecha_pago DESC);

-- Caja
CREATE INDEX IF NOT EXISTS idx_sesiones_caja_activas 
    ON sesiones_caja(caja_id, esta_abierta, fecha_apertura DESC) 
    WHERE esta_abierta = TRUE;

CREATE INDEX IF NOT EXISTS idx_sesiones_caja_usuario 
    ON sesiones_caja(usuario_apertura_id, fecha_apertura DESC);

CREATE INDEX IF NOT EXISTS idx_movimientos_caja_sesion 
    ON movimientos_caja(sesion_caja_id, tipo_movimiento, creado_en);

CREATE INDEX IF NOT EXISTS idx_movimientos_caja_pedido 
    ON movimientos_caja(pedido_id) 
    WHERE pedido_id IS NOT NULL;

-- Producción
CREATE INDEX IF NOT EXISTS idx_planes_produccion_sede_fecha 
    ON planes_produccion(sede_id, fecha_produccion, estado);

CREATE INDEX IF NOT EXISTS idx_planes_produccion_activos 
    ON planes_produccion(tienda_id, estado) 
    WHERE estado IN ('confirmado', 'en_proceso');

CREATE INDEX IF NOT EXISTS idx_detalles_plan_producto 
    ON detalles_plan_produccion(plan_id, producto_id, estado);

CREATE INDEX IF NOT EXISTS idx_detalles_plan_personalizados 
    ON detalles_plan_produccion(plan_id, es_personalizado) 
    WHERE es_personalizado = TRUE;

CREATE INDEX IF NOT EXISTS idx_detalles_plan_pedido_cliente 
    ON detalles_plan_produccion(pedido_cliente_id) 
    WHERE pedido_cliente_id IS NOT NULL;

CREATE INDEX IF NOT EXISTS idx_detalles_plan_detalle_pedido 
    ON detalles_plan_produccion(detalle_pedido_id) 
    WHERE detalle_pedido_id IS NOT NULL;

CREATE INDEX IF NOT EXISTS idx_recetas_producto 
    ON recetas(producto_id, insumo_id);

CREATE INDEX IF NOT EXISTS idx_conteos_diarios_sede_fecha 
    ON conteos_diarios(sede_id, fecha_conteo DESC);

CREATE INDEX IF NOT EXISTS idx_detalle_conteo 
    ON detalle_conteo_diario(conteo_id, producto_id);

-- Inventario (Insumos)
CREATE INDEX IF NOT EXISTS idx_inventario_insumos_tienda 
    ON inventario_insumos_sedes(tienda_id, cantidad_actual) 
    WHERE cantidad_actual > 0;

CREATE INDEX IF NOT EXISTS idx_movimientos_insumos_sede_fecha 
    ON movimientos_inventario_insumos(sede_id, insumo_id, creado_en DESC);

CREATE INDEX IF NOT EXISTS idx_movimientos_insumos_tipo 
    ON movimientos_inventario_insumos(tienda_id, tipo_movimiento, creado_en DESC);

CREATE INDEX IF NOT EXISTS idx_movimientos_insumos_orden_compra 
    ON movimientos_inventario_insumos(orden_compra_id) 
    WHERE orden_compra_id IS NOT NULL;

CREATE INDEX IF NOT EXISTS idx_movimientos_insumos_plan 
    ON movimientos_inventario_insumos(plan_produccion_id) 
    WHERE plan_produccion_id IS NOT NULL;

CREATE INDEX IF NOT EXISTS idx_movimientos_insumos_transferencia 
    ON movimientos_inventario_insumos(transferencia_id) 
    WHERE transferencia_id IS NOT NULL;

-- Inventario (Productos)
CREATE INDEX IF NOT EXISTS idx_inventario_productos_disponibles 
    ON inventario_productos(tienda_id, cantidad_actual) 
    WHERE cantidad_actual > 0;

CREATE INDEX IF NOT EXISTS idx_movimientos_productos_sede_fecha 
    ON movimientos_inventario_productos(sede_id, producto_id, creado_en DESC);

CREATE INDEX IF NOT EXISTS idx_movimientos_productos_tipo 
    ON movimientos_inventario_productos(tienda_id, tipo_movimiento, creado_en DESC);

CREATE INDEX IF NOT EXISTS idx_movimientos_productos_pedido 
    ON movimientos_inventario_productos(pedido_id) 
    WHERE pedido_id IS NOT NULL;

CREATE INDEX IF NOT EXISTS idx_movimientos_productos_plan 
    ON movimientos_inventario_productos(plan_produccion_id) 
    WHERE plan_produccion_id IS NOT NULL;

-- Transferencias
CREATE INDEX IF NOT EXISTS idx_transferencias_origen 
    ON transferencias_inventario(sede_origen_id, estado, fecha_solicitud DESC);

CREATE INDEX IF NOT EXISTS idx_transferencias_destino 
    ON transferencias_inventario(sede_destino_id, estado, fecha_solicitud DESC);

CREATE INDEX IF NOT EXISTS idx_items_transferencia 
    ON items_transferencia(transferencia_id);

-- Compras
CREATE INDEX IF NOT EXISTS idx_proveedores_tienda_activos 
    ON proveedores(tienda_id, activo) 
    WHERE activo = TRUE;

CREATE INDEX IF NOT EXISTS idx_ordenes_compra_estado 
    ON ordenes_compra(tienda_id, estado, fecha_emision DESC);

CREATE INDEX IF NOT EXISTS idx_ordenes_compra_sede 
    ON ordenes_compra(sede_destino_id, estado);

CREATE INDEX IF NOT EXISTS idx_ordenes_compra_proveedor 
    ON ordenes_compra(proveedor_id, fecha_emision DESC);

CREATE INDEX IF NOT EXISTS idx_detalles_orden_compra 
    ON detalles_orden_compra(orden_compra_id, insumo_id);

CREATE INDEX IF NOT EXISTS idx_detalles_orden_pendientes 
    ON detalles_orden_compra(orden_compra_id, recibido_completo) 
    WHERE recibido_completo = FALSE;

-- Catálogo
CREATE INDEX IF NOT EXISTS idx_productos_tienda_activos 
    ON productos(tienda_id, activo) 
    WHERE activo = TRUE;

CREATE INDEX IF NOT EXISTS idx_productos_categoria 
    ON productos(categoria_id, activo);

CREATE INDEX IF NOT EXISTS idx_productos_pos_visible 
    ON productos(tienda_id, visible_en_pos) 
    WHERE visible_en_pos = TRUE AND activo = TRUE;

CREATE INDEX IF NOT EXISTS idx_productos_storefront_visible 
    ON productos(tienda_id, visible_en_storefront) 
    WHERE visible_en_storefront = TRUE AND activo = TRUE;

CREATE INDEX IF NOT EXISTS idx_productos_destacados 
    ON productos(tienda_id, destacado_storefront) 
    WHERE destacado_storefront = TRUE AND activo = TRUE;

CREATE INDEX IF NOT EXISTS idx_productos_busqueda 
    ON productos(tienda_id, slug, sku);

CREATE INDEX IF NOT EXISTS idx_categorias_tienda_activas 
    ON categorias(tienda_id, activa, orden_visual) 
    WHERE activa = TRUE;

-- Insumos
CREATE INDEX IF NOT EXISTS idx_insumos_tienda_activos 
    ON insumos(tienda_id, activo) 
    WHERE activo = TRUE;

CREATE INDEX IF NOT EXISTS idx_insumos_stock_bajo 
    ON insumos(tienda_id, stock_actual_global, stock_minimo_global) 
    WHERE stock_actual_global <= stock_minimo_global AND activo = TRUE;

-- Facturación Tienda
CREATE INDEX IF NOT EXISTS idx_tienda_comprobantes_pedido 
    ON tienda_comprobantes(pedido_id);

CREATE INDEX IF NOT EXISTS idx_tienda_comprobantes_serie 
    ON tienda_comprobantes(tipo_comprobante, serie_id, correlativo DESC);

CREATE INDEX IF NOT EXISTS idx_tienda_comprobantes_fecha 
    ON tienda_comprobantes(tienda_id, fecha_emision DESC);

CREATE INDEX IF NOT EXISTS idx_tienda_comprobantes_sunat 
    ON tienda_comprobantes(estado_sunat, fecha_emision DESC) 
    WHERE estado_sunat IN ('pendiente', 'observado', 'rechazado');

CREATE INDEX IF NOT EXISTS idx_tienda_series_activas 
    ON tienda_series(tienda_id, sede_id, activa) 
    WHERE activa = TRUE;

-- Configuración & Seguridad (Usuarios y Roles)
CREATE INDEX IF NOT EXISTS idx_usuarios_tienda_rol 
    ON usuarios_tienda(rol_id, activo) 
    WHERE activo = TRUE;

CREATE INDEX IF NOT EXISTS idx_usuarios_tienda_correo 
    ON usuarios_tienda(tienda_id, correo) 
    WHERE eliminado_en IS NULL;

CREATE INDEX IF NOT EXISTS idx_usuario_sedes 
    ON usuario_sedes(usuario_id, es_sede_principal);

CREATE INDEX IF NOT EXISTS idx_usuario_sedes_por_sede 
    ON usuario_sedes(sede_id, es_sede_principal DESC);

CREATE INDEX IF NOT EXISTS idx_roles_tienda 
    ON roles(tienda_id) 
    WHERE es_sistema = FALSE;

CREATE INDEX IF NOT EXISTS idx_roles_permisos_permiso 
    ON roles_permisos(permiso_id);

CREATE INDEX IF NOT EXISTS idx_permisos_modulo 
    ON permisos(modulo, slug);

-- Auditoría
CREATE INDEX IF NOT EXISTS idx_auditoria_tienda_fecha 
    ON auditoria_usuarios(tienda_id, creado_en DESC);

CREATE INDEX IF NOT EXISTS idx_auditoria_usuario 
    ON auditoria_usuarios(usuario_id, creado_en DESC) 
    WHERE usuario_id IS NOT NULL;

CREATE INDEX IF NOT EXISTS idx_auditoria_entidad 
    ON auditoria_usuarios(entidad, entidad_id, creado_en DESC);

-- CMS (Storefront)
CREATE INDEX IF NOT EXISTS idx_paginas_storefront_activas 
    ON paginas_storefront(tienda_id, activa, orden_menu) 
    WHERE activa = TRUE;

CREATE INDEX IF NOT EXISTS idx_paginas_storefront_menu 
    ON paginas_storefront(tienda_id, visible_en_menu, orden_menu) 
    WHERE visible_en_menu = TRUE AND activa = TRUE;

-- ==============================
-- ÍNDICES ESPECIALES PARA REPORTES
-- ==============================

-- Ventas por período
CREATE INDEX IF NOT EXISTS idx_pedidos_reporte_ventas 
    ON pedidos(tienda_id, estado_pedido, creado_en) 
    WHERE estado_pedido IN ('pagado', 'en_preparacion', 'listo_entrega', 'entregado');

-- Productos más vendidos
CREATE INDEX IF NOT EXISTS idx_detalles_pedido_stats 
    ON detalles_pedido(producto_id, cantidad);

-- Análisis de merma
CREATE INDEX IF NOT EXISTS idx_detalles_plan_merma 
    ON detalles_plan_produccion(producto_id, cantidad_merma) 
    WHERE cantidad_merma > 0;

-- Eficiencia de producción
CREATE INDEX IF NOT EXISTS idx_planes_produccion_tiempos 
    ON planes_produccion(tienda_id, fecha_produccion, hora_inicio_real, hora_fin_real) 
    WHERE estado = 'finalizado';

-- ==============================
-- ÍNDICES DE BÚSQUEDA FULL-TEXT (Opcional)
-- ==============================

-- Para búsqueda de productos por nombre/descripción
CREATE INDEX IF NOT EXISTS idx_productos_busqueda_texto 
    ON productos USING gin(to_tsvector('spanish', nombre || ' ' || COALESCE(descripcion, '')));

-- Para búsqueda de clientes
CREATE INDEX IF NOT EXISTS idx_clientes_busqueda_texto 
    ON clientes USING gin(to_tsvector('spanish', nombre_doc || ' ' || COALESCE(email::text, '')));

-- ==============================
-- ÍNDICES COMPUESTOS PARA CONSULTAS COMPLEJAS
-- ==============================

-- Dashboard de caja diaria
CREATE INDEX IF NOT EXISTS idx_sesiones_caja_dashboard 
    ON sesiones_caja(caja_id, fecha_apertura::date, esta_abierta);

-- Pedidos pendientes por sede y estado
CREATE INDEX IF NOT EXISTS idx_pedidos_pendientes_sede 
    ON pedidos(sede_origen_id, estado_pedido, estado_pago, fecha_entrega_pactada) 
    WHERE estado_pedido NOT IN ('entregado', 'cancelado');

-- Stock crítico por sede
CREATE INDEX IF NOT EXISTS idx_inventario_productos_critico 
    ON inventario_productos(sede_id, cantidad_actual);

-- Órdenes de compra pendientes de recepción
CREATE INDEX IF NOT EXISTS idx_ordenes_compra_pendientes 
    ON ordenes_compra(sede_destino_id, estado, fecha_recepcion_esperada) 
    WHERE estado IN ('enviada', 'recibida_parcial');

-- ==============================
-- COMENTARIOS EN ÍNDICES CLAVE
-- ==============================

COMMENT ON INDEX idx_pedidos_borrador_pos IS 
'Índice para carritos temporales del POS. Permite al vendedor retomar pedidos en borrador';

COMMENT ON INDEX idx_productos_storefront_visible IS 
'Optimiza la carga del catálogo público del storefront';

COMMENT ON INDEX idx_planes_produccion_sede_fecha IS 
'Índice crítico para el módulo de producción diaria';

COMMENT ON INDEX idx_detalles_plan_personalizados IS 
'Ayuda al panadero a identificar rápidamente productos personalizados en el plan';

COMMENT ON INDEX idx_insumos_stock_bajo IS 
'Alertas automáticas de stock bajo para generar órdenes de compra';

-- =================================
-- TRIGGERS PARA ACTUALIZAR actualizado_en
-- =================================

CREATE OR REPLACE FUNCTION update_actualizado_en()
RETURNS TRIGGER AS $
BEGIN
    NEW.actualizado_en = CURRENT_TIMESTAMP;
RETURN NEW;
END;
$ LANGUAGE plpgsql;

CREATE TRIGGER trigger_usuarios_superadmin_actualizado
    BEFORE UPDATE ON usuarios_superadmin
    FOR EACH ROW
    EXECUTE FUNCTION update_actualizado_en();

CREATE TRIGGER trigger_tiendas_actualizado
    BEFORE UPDATE ON tiendas
    FOR EACH ROW
    EXECUTE FUNCTION update_actualizado_en();

CREATE TRIGGER trigger_sedes_actualizado
    BEFORE UPDATE ON sedes
    FOR EACH ROW
    EXECUTE FUNCTION update_actualizado_en();

CREATE TRIGGER trigger_dominios_tienda_actualizado
    BEFORE UPDATE ON dominios_tienda
    FOR EACH ROW
    EXECUTE FUNCTION update_actualizado_en();

CREATE TRIGGER trigger_planes_actualizado
    BEFORE UPDATE ON planes
    FOR EACH ROW
    EXECUTE FUNCTION update_actualizado_en();

CREATE TRIGGER trigger_suscripciones_actualizado
    BEFORE UPDATE ON suscripciones
    FOR EACH ROW
    EXECUTE FUNCTION update_actualizado_en();

CREATE TRIGGER trigger_series_actualizado
    BEFORE UPDATE ON series
    FOR EACH ROW
    EXECUTE FUNCTION update_actualizado_en();

CREATE TRIGGER trigger_comprobantes_actualizado
    BEFORE UPDATE ON comprobantes
    FOR EACH ROW
    EXECUTE FUNCTION update_actualizado_en();

CREATE TRIGGER trigger_tickets_soporte_actualizado
    BEFORE UPDATE ON tickets_soporte
    FOR EACH ROW
    EXECUTE FUNCTION update_actualizado_en();

CREATE TRIGGER trigger_usuarios_tienda_actualizado
    BEFORE UPDATE ON usuarios_tienda
    FOR EACH ROW
    EXECUTE FUNCTION update_actualizado_en();

CREATE TRIGGER trigger_productos_actualizado
    BEFORE UPDATE ON productos
    FOR EACH ROW
    EXECUTE FUNCTION update_actualizado_en();

CREATE TRIGGER trigger_ordenes_compra_actualizado
    BEFORE UPDATE ON ordenes_compra
    FOR EACH ROW
    EXECUTE FUNCTION update_actualizado_en();

CREATE TRIGGER trigger_clientes_actualizado
    BEFORE UPDATE ON clientes
    FOR EACH ROW
    EXECUTE FUNCTION update_actualizado_en();

CREATE TRIGGER trigger_pedidos_actualizado
    BEFORE UPDATE ON pedidos
    FOR EACH ROW
    EXECUTE FUNCTION update_actualizado_en();

CREATE TRIGGER trigger_movimientos_caja_actualizado
    BEFORE UPDATE ON movimientos_caja
    FOR EACH ROW
    EXECUTE FUNCTION update_actualizado_en();

CREATE TRIGGER trigger_stock_ideal_actualizado
    BEFORE UPDATE ON stock_ideal
    FOR EACH ROW
    EXECUTE FUNCTION update_actualizado_en();

CREATE TRIGGER trigger_planes_produccion_actualizado
    BEFORE UPDATE ON planes_produccion
    FOR EACH ROW
    EXECUTE FUNCTION update_actualizado_en();

CREATE TRIGGER trigger_inventario_insumos_sedes_actualizado
    BEFORE UPDATE ON inventario_insumos_sedes
    FOR EACH ROW
    EXECUTE FUNCTION update_actualizado_en();

CREATE TRIGGER trigger_inventario_productos_actualizado
    BEFORE UPDATE ON inventario_productos
    FOR EACH ROW
    EXECUTE FUNCTION update_actualizado_en();

CREATE TRIGGER trigger_configuracion_tienda_actualizado
    BEFORE UPDATE ON configuracion_tienda
    FOR EACH ROW
    EXECUTE FUNCTION update_actualizado_en();

CREATE TRIGGER trigger_paginas_storefront_actualizado
    BEFORE UPDATE ON paginas_storefront
    FOR EACH ROW
    EXECUTE FUNCTION update_actualizado_en();